package filesystems

import (
	"context"
	"fmt"
	"net/http"

	"github.com/hashicorp/go-azure-sdk/sdk/client"
	"github.com/hashicorp/go-azure-sdk/sdk/odata"
)

type CreateInput struct {
	// A map of base64-encoded strings to store as user-defined properties with the File System
	// Note that items may only contain ASCII characters in the ISO-8859-1 character set.
	// This automatically gets converted to a comma-separated list of name and
	// value pairs before sending to the API
	Properties map[string]string
}

type CreateResponse struct {
	HttpResponse *client.Response
}

// Create creates a Data Lake Store Gen2 FileSystem within a Storage Account
func (c Client) Create(ctx context.Context, fileSystemName string, input CreateInput) (resp CreateResponse, err error) {

	if fileSystemName == "" {
		return resp, fmt.Errorf("`fileSystemName` cannot be an empty string")
	}

	opts := client.RequestOptions{
		ContentType: "application/xml; charset=utf-8",
		ExpectedStatusCodes: []int{
			http.StatusCreated,
		},
		HttpMethod: http.MethodPut,
		OptionsObject: createOptions{
			properties: input.Properties,
		},

		Path: fmt.Sprintf("/%s", fileSystemName),
	}

	req, err := c.Client.NewRequest(ctx, opts)

	if err != nil {
		err = fmt.Errorf("building request: %+v", err)
		return
	}
	resp.HttpResponse, err = req.Execute(ctx)
	if err != nil {
		err = fmt.Errorf("executing request: %+v", err)
		return
	}

	return
}

type createOptions struct {
	properties map[string]string
}

func (o createOptions) ToHeaders() *client.Headers {
	headers := &client.Headers{}
	props := buildProperties(o.properties)
	if props != "" {
		headers.Append("x-ms-properties", props)
	}

	return headers
}

func (createOptions) ToOData() *odata.Query {
	return nil
}

func (createOptions) ToQuery() *client.QueryParams {
	return fileSystemOptions{}.ToQuery()
}
